const byId = (id) => document.getElementById(id);

const statusEl = byId("license-status");
const idEl = byId("license-id");
const customerEl = byId("license-customer");
const modeEl = byId("license-mode");
const expiresEl = byId("license-expires");
const authorityEl = byId("license-authority");
const registryEl = byId("license-registry");

function formatEpoch(epoch) {
  if (!epoch || Number.isNaN(Number(epoch))) {
    return "--";
  }
  const date = new Date(Number(epoch) * 1000);
  return date.toISOString();
}

function normalizeMode(data) {
  const mode = (data?.mode || data?.license_type || "").toUpperCase();
  if (mode === "EVIDENCE_ONLY" || mode === "EVIDENCE") {
    return "Evidence Mode";
  }
  if (mode === "AUTHORITY" || mode === "DECISION_AUTHORITY") {
    return "Decision Authority Mode";
  }
  return mode || "--";
}

function setText(el, value) {
  if (!el) return;
  el.textContent = value || "--";
}

async function loadLicense() {
  try {
    const res = await fetch("/api/license/status");
    if (!res.ok) {
      throw new Error(`status ${res.status}`);
    }
    const data = await res.json();
    const valid = data?.valid === true;
    const permissions = data?.permissions || {};

    setText(statusEl, valid ? "Valid" : "Invalid");
    setText(idEl, data?.license_id || "--");
    setText(customerEl, data?.customer_id || "--");
    setText(modeEl, normalizeMode(data));
    setText(expiresEl, formatEpoch(data?.expires_at_epoch));
    setText(authorityEl, permissions.issue_decisions ? "Enabled" : "Disabled");
  } catch (err) {
    setText(statusEl, "Unavailable");
    setText(idEl, "--");
    setText(customerEl, "--");
    setText(modeEl, "--");
    setText(expiresEl, "--");
    setText(authorityEl, "--");
  }
}

async function loadRegistry() {
  try {
    const res = await fetch("/api/registry");
    if (!res.ok) {
      throw new Error(`registry ${res.status}`);
    }
    const data = await res.json();
    const version = data?.registry_version || "unknown";
    setText(registryEl, `${version} (Locked)`);
  } catch (err) {
    setText(registryEl, "--");
  }
}

async function init() {
  await Promise.all([loadLicense(), loadRegistry()]);
}

document.addEventListener("DOMContentLoaded", init);
